
package dk.clondyke.fsxveg;

import java.io.File;
import java.util.HashMap;
import java.util.prefs.Preferences;

public class FSXVegModel {
	private String programName = "FSX Vegetation Renaming Tool";
	private Preferences prefs;

	private static FSXVegModel instance = null;
	
	public static final int SUMMER = 1;
	public static final int WINTER = 2;
	public static final int HARDWINTER = 3;
	public static final int SPRING = 4;
	public static final int FALL = 5;

	public static final int MIXED = 6;
	public static final int NONE = 7;
	public static final int EXCEPTION = 8;

	private int state = NONE;
	
	private String exceptionMsg = "";

	private HashMap fileSets = new HashMap();
	private int summers = 0;
	private int winters = 0;
	private int hardWinters = 0;
	private int springs = 0;
	private int falls = 0;
	private int mixeds = 0;
	private String seasonsAsString = "";
	
	private FSXVegModel() {
		prefs = Preferences.userNodeForPackage(this.getClass());
		//prefs.put("scenery_path", "");
		File testDir = new File("Scenery");
		if (testDir.exists() && testDir.isDirectory()) {
			prefs.put("scenery_path", testDir.getAbsolutePath());
		}
		
		Log.setEnabled(false);
		Log.init();
		Log.log("GENERAL SYSTEM INFORMATION:");
		Log.log("Operating system: " + System.getProperty("os.name"));
		Log.log("Architecture: " + System.getProperty("os.arch"));
		Log.log("Version: " + System.getProperty("os.version"));
		Log.log("Java vendor: " + System.getProperty("java.vendor"));
		Log.log("Java version: " + System.getProperty("java.version"));
		Log.log("");
		
		
		
		readFileSets();
	}

	public static FSXVegModel getInstance() {
		if (instance == null) {
			instance = new FSXVegModel(); 
		}
		return instance;
	}

	public int getState() {
		return state;
	}
	
	public String getActiveSeasonsAsString() {
		return seasonsAsString;
	}
	
	private void readFileSets() {
		fileSets = new HashMap();
		summers = 0;
		winters = 0;
		hardWinters = 0;
		springs = 0;
		falls = 0;
		mixeds = 0;

		state = MIXED;		

		Log.log("Starting to read filesets ....");
		
		if (!getSceneryPath().equals("")) {

			Log.log("Finding files in directory: " + getSceneryPath());
			
			File dir = new File(getSceneryPath());
			if (!dir.exists()) {
				state = EXCEPTION;
				exceptionMsg = "Directory does not exist.";
				Log.log(exceptionMsg);
				return;
			}

			
			File[] files = dir.listFiles();

			Log.log("There are " + files.length + " in the directory.");
			Log.log("The program will now test all file names to find a summer file.");
			
			for (int i = 0; i < files.length; i++) {
				Log.log(i + ": " + files[i].getName());

				if (files[i].getName().length() <= 7) {
					Log.log("\tFilename too short to be considered.");
					continue;
				}
				String start = files[i].getName().toLowerCase().substring(0, files[i].getName().length()-7);
				String end = files[i].getName().toLowerCase().substring(files[i].getName().length()-7);

				Log.log("\tStart part of filename:\t" + start);
				Log.log("\tEnd part of filename:\t" + end);

				SeasonalFileSet fileSet = null;
				
				if (end.substring(0, 4).equals("_su.")) {
					//System.out.println(files[i].getName());
					Log.log("\tThis is a summer file.");

					if (fileSets.get(start) != null) {
						fileSet = (SeasonalFileSet) fileSets.get(start); 
						Log.log("\tA fileset was already found: \"" + start + "\".");
					} else {
						SeasonalFileSet newSet = new SeasonalFileSet();
						fileSets.put(start, newSet);
						//System.out.println("Putting: " + start);
						fileSet = newSet;
						Log.log("\tCreating a new fileset: \"" + start + "\".");
					}
					
					if (fileSet != null) {
						if (end.substring(0, 4).equals("_su."))
							fileSet.setSummerFile(files[i]);
					}
				}
			}

			Log.log("The program will now test all file names to find other possible seasonal files.");
			
			// Making the loop for the rest of the files
			for (int i = 0; i < files.length; i++) {
				Log.log(i + ": " + files[i].getName());

				if (files[i].getName().length() <= 7)
					continue;

				String start = files[i].getName().toLowerCase().substring(0, files[i].getName().length()-7);
				String end = files[i].getName().toLowerCase().substring(files[i].getName().length()-7);
				
				
				SeasonalFileSet fileSet = null;

				
				if (fileSets.get(start) != null
						&& (end.substring(0, 4).equals("_hw.")
							|| end.substring(0, 4).equals("_wi.")
							|| end.substring(0, 4).equals("_sp.")
							|| end.substring(0, 4).equals("_fa."))) {
					fileSet = (SeasonalFileSet) fileSets.get(start); 

					if (end.substring(0, 4).equals("_hw.")) {
						fileSet.setHardWinterFile(files[i]);
						Log.log("\tSetting " + files[i].getName() + " as the hard winter file for fileset: \"" + start + "\".");
					}
					if (end.substring(0, 4).equals("_wi.")) {
						fileSet.setWinterFile(files[i]);
						Log.log("\tSetting " + files[i].getName() + " as the winter file for fileset: \"" + start + "\".");
					}
					if (end.substring(0, 4).equals("_sp.")) {
						fileSet.setSpringFile(files[i]);
						Log.log("\tSetting " + files[i].getName() + " as the spring file for fileset: \"" + start + "\".");
					}
					if (end.substring(0, 4).equals("_fa.")) {
						fileSet.setFallFile(files[i]);
						Log.log("\tSetting " + files[i].getName() + " as the fall file for fileset: \"" + start + "\".");
					}
				}
			}
			
			
			// Making the loop again to find possible default files for the sets that are now created
			for (int i = 0; i < files.length; i++) {
				if (files[i].getName().length() <= 7)
					continue;

				String keyName = files[i].getName().substring(0, files[i].getName().length()-4);
				String end = files[i].getName().substring(files[i].getName().length()-4);
			
				SeasonalFileSet fileSet = (SeasonalFileSet) fileSets.get(keyName.toLowerCase());
				if (fileSet != null && end.equals(".bgl")) {
					//System.out.println(keyName + ": " + fileSet);
					//System.out.println(end.toLowerCase());
					fileSet.setDefaultFile(files[i]);

				
					Log.log("\tSetting " + files[i].getName() + " as the default file for set \"" + keyName + "\".");
				
				}
			}
		
		}

	
		Object keys[] = fileSets.keySet().toArray();
		
		for (int i = 0; i < keys.length; i++) {
			//System.out.println(keys[i]);
			
			SeasonalFileSet fileSet = (SeasonalFileSet) fileSets.get(keys[i]);
			
			int season = fileSet.getActiveSeason();
			//System.out.println(season);
			if (season == SUMMER)
				summers++;
			if (season == WINTER)
				winters++;
			if (season == HARDWINTER)
				hardWinters++;
			if (season == SPRING)
				springs++;
			if (season == FALL)
				falls++;
			if (season == MIXED)
				mixeds++;
			
			//System.out.println("Is complete: " + fileSet.isComplete());
			
		}
		
		if (summers == fileSets.size()) {
			state = SUMMER;
		}
		if (winters == fileSets.size()) {
			state = WINTER;
		}
		if (hardWinters == fileSets.size()) {
			state = HARDWINTER;
		}
		if (falls == fileSets.size()) {
			state = FALL;
		}
		if (springs == fileSets.size()) {
			state = SPRING;
		}
	
		if (summers == 0 && winters == 0 && hardWinters == 0 && springs == 0 && falls == 0) {
			state = NONE;
		}

		if ((summers > 0 && winters > 0) || mixeds > 0) {
			state = MIXED;
		}

		seasonsAsString = "";
		if (summers > 0)
			seasonsAsString = "summer";
		if (winters > 0) {
			if (!seasonsAsString.equals(""))
				seasonsAsString += " + ";
			seasonsAsString += "winter";
		}

		if (hardWinters > 0) {
			if (!seasonsAsString.equals(""))
				seasonsAsString += " + ";
			seasonsAsString += "hard winter";
		}

		if (springs > 0) {
			if (!seasonsAsString.equals(""))
				seasonsAsString += " + ";
			seasonsAsString += "spring";
		}

		if (falls > 0) {
			if (!seasonsAsString.equals(""))
				seasonsAsString += " + ";
			seasonsAsString += "fall";
		}
		
		/*
		System.out.println("Summers: " + summers);
		System.out.println("Winters: " + winters);
		System.out.println("Hard winters: " + hardWinters);
		System.out.println("Springs: " + springs);
		System.out.println("Falls: " + falls);
		System.out.println("Mixeds: " + mixeds);
		*/
	
	}
	
	public void setActiveSeason(int season) {

		Object keys[] = fileSets.keySet().toArray();
		
		boolean success = true;
		
		for (int i = 0; i < keys.length; i++) {
			SeasonalFileSet fileSet = (SeasonalFileSet) fileSets.get(keys[i]);

			//System.out.println("Setting season for " + keys[i]);
			
			if (!fileSet.setActiveSeason(season)) {
				state = EXCEPTION;
				exceptionMsg = "One or more files could not be renamed.";
				success = false;
				break;
			}
		}		

		if (success) {
			readFileSets();
		}
	}
	
	public String getProgramName() {
		return programName;
	}

	public String getExceptionMessage() {
		return exceptionMsg;
	}

	public boolean atLeastOneSetHasSuffix() {
		
		Object keys[] = fileSets.keySet().toArray();
		
		boolean suffixFound = false;
		for (int i = 0; i < keys.length; i++) {
			SeasonalFileSet fileSet = (SeasonalFileSet) fileSets.get(keys[i]);
			if (fileSet.defaultFileHasSuffix()) {
				suffixFound = true;
				break;
			}
		}		
		return suffixFound;
	}
	
	public String getFilesToString() {
		String files = "";

		Object keys[] = fileSets.keySet().toArray();
		
		for (int i = 0; i < keys.length; i++) {
			SeasonalFileSet fileSet = (SeasonalFileSet) fileSets.get(keys[i]);
			if (fileSet.getDefaultFile() != null) {
				files += fileSet.getDefaultFile().getName() + "\r\n";
			}
			if (fileSet.getSummerFile() != null) {
				files += fileSet.getSummerFile().getName() + "\r\n";
			}
			if (fileSet.getWinterFile() != null) {
				files += fileSet.getWinterFile().getName() + "\r\n";
			}
			if (fileSet.getHardWinterFile() != null) {
				files += fileSet.getHardWinterFile().getName() + "\r\n";
			}
			if (fileSet.getSpringFile() != null) {
				files += fileSet.getSpringFile().getName() + "\r\n";
			}
			if (fileSet.getFallFile() != null) {
				files += fileSet.getFallFile().getName() + "\r\n";
			}
		}		
	
		return files.trim();
	}
	
	public String getSceneryPath() {
		return prefs.get("scenery_path", "");
	}

	public void setSceneryPath(String path) {
		if (path.length() > 7 && !path.substring(path.length() - 7).toLowerCase().equals("scenery")) {
			File testDir = new File(path + "\\Scenery");
			if (testDir.exists())
				path = testDir.getAbsolutePath();
		}
		prefs.put("scenery_path", path);
		readFileSets();
	}
}
